// ===============================================================
// grouping config.js
//  - 3グループ固定
//  - 配置場所 select：Label(表示)フィールドのみ
//  - 対象フィールド：全フィールドをチェックで選択
//  - トグル開閉はクリック委譲で一括管理
//  - /fields APIは1回だけ叩いてキャッシュ
// ===============================================================
(function () {
  'use strict';

  var config = atPocket.plugin.app.getConfig();

  // config.values と直下の両対応
  function cfg(key){
    return (config.values && config.values[key] !== undefined)
      ? config.values[key]
      : config[key];
  }

  // 保存値復元（CSV/JSON/配列 全対応）
  function parseSaved(raw){
    if (!raw) return [];
    if (Object.prototype.toString.call(raw) === "[object Array]") {
      return raw.filter(Boolean).map(String);
    }
    if (typeof raw === "string") {
      try {
        var arr = JSON.parse(raw);
        if (Object.prototype.toString.call(arr) === "[object Array]") {
          return arr.filter(Boolean).map(String);
        }
      } catch(e){}
      return raw.split(",").map(function(s){ return s.trim(); }).filter(Boolean);
    }
    return [];
  }

  // ---------------------------
  // クリック委譲：対象選択トグルを一括管理
  // ---------------------------
  document.addEventListener('click', function(ev){
    var t = ev.target;

    // toggle_group_fields or toggle_group_fields_N
    var btn = t.closest ? t.closest('button[id^="toggle_group_fields"]') : null;
    if (!btn) return;

    var id = btn.id;
    var m = id.match(/^toggle_group_fields_(\d+)$/);

    var box = null;
    if (m) {
      box = document.getElementById('group_fields_container_' + m[1]);
    }
    if (!box) return;

    var cur = getComputedStyle(box).display;
    box.style.display = (cur === 'none') ? 'block' : 'none';
    btn.textContent = (cur === 'none') ? '▲ 対象選択' : '▼ 対象選択';
  });

  // ---------------------------
  // hiddenへ同期
  // ---------------------------
  function syncHidden(container, hiddenEl){
    var checked = container.querySelectorAll('input[type=checkbox]:checked');
    var ids = [];
    for (var i=0; i<checked.length; i++){
      ids.push(String(checked[i].value));
    }
    hiddenEl.value = ids.join(',');
    if (config.values) config.values[hiddenEl.name || hiddenEl.id] = hiddenEl.value;
  }

// ---------------------------
// 対象フィールド：全フィールドチェック描画
//   - ★保存は uniqueId（条件付きプラグインと同じ）
// ---------------------------
function renderAllFieldCheckboxes(result){
const excludeIds = [-101, -104, -105, -106, -107, -119];
  const fields = result.fields.filter(f => !excludeIds.includes(f.fieldId));


  [1,2,3].forEach(function(n){
    var container = document.getElementById('group_fields_container_' + n);
    var hiddenEl  = document.getElementById('group_fields_' + n);
    if (!container || !hiddenEl) return;

    // 保存値（uniqueIdのCSV）を復元
    var saved = parseSaved(cfg('group_fields_' + n));
    var savedSet = {};
    saved.forEach(function(id){ savedSet[id] = true; });

    container.innerHTML = '';

    // ★ 全フィールド分チェックボックスを描画
    fields.forEach(function(field){
      var id = String(field.uniqueId);  // ← uniqueId を使う（nameチェックしない）

      var label = document.createElement('label');
      label.style.display = 'inline-block';
      label.style.marginRight = '14px';
      label.style.minWidth = '180px';

      var cb = document.createElement('input');
      cb.type = 'checkbox';
      cb.value = id;
      cb.checked = !!savedSet[id];
            cb.addEventListener('click', function () {
        if (cb.checked) {
          // 他グループで既に選ばれていたら、このグループのチェックを無効にする
          if (isFieldUsedInOtherGroups(n, cb.value)) {
            cb.checked = false;            // 今押した方（例: グループ2）を元に戻す
            syncHidden(container, hiddenEl); 
            alert('この項目は他のグループで既に選択されています。');
            return;
          }
        }
        // 通常の同期
        syncHidden(container, hiddenEl);
      });



      // 表示名は caption / name / label のどれか
      var text = field.caption || field.name || field.label || ('ID:' + id);
      label.appendChild(cb);
      label.appendChild(document.createTextNode(' ' + text));
      container.appendChild(label);
    });

    // hidden初期反映
    hiddenEl.value = saved.join(',');

    // 全選択 / 全解除
    var btnWrap = document.createElement('div');
    btnWrap.style.marginTop = '10px';
    btnWrap.style.display = 'flex';
    btnWrap.style.gap = '6px';

        var allBtn = document.createElement('button');
    allBtn.type = 'button';
    allBtn.textContent = '全選択';
    allBtn.onclick = function () {
      var cbs = container.querySelectorAll('input[type=checkbox]');
      var hasConflict = false;

      // まず「重複があるかどうか」だけチェック（この時点では何も変更しない）
      for (var i = 0; i < cbs.length; i++) {
        var cb = cbs[i];
        if (cb.checked) continue; // すでにONなら対象外

        if (isFieldUsedInOtherGroups(n, cb.value)) {
          hasConflict = true;
          break;
        }
      }

      // 1つでも重複があれば、何も変更せずアラートだけ出して終了
      if (hasConflict) {
        alert('他のグループで既に選択されている項目があるため、「全選択」は実行できません。');
        return;
      }

      // 重複がない場合だけ、全て選択状態にする
      for (var i = 0; i < cbs.length; i++) {
        var cb2 = cbs[i];
        if (!cb2.checked) {
          cb2.checked = true;
        }
      }
      syncHidden(container, hiddenEl);
    };

    var noneBtn = document.createElement('button');
    noneBtn.type = 'button';
    noneBtn.textContent = '全解除';
    noneBtn.onclick = function(){
      var cbs = container.querySelectorAll('input[type=checkbox]');
      for (var i=0; i<cbs.length; i++) cbs[i].checked = false;
      syncHidden(container, hiddenEl);
    };

    btnWrap.appendChild(allBtn);
    btnWrap.appendChild(noneBtn);
    container.appendChild(btnWrap);
  });
}

  // セレクトボックスオプション生成
  function createSelectBox(result) {

    // 連結先フィールドにセットするテキストタイプのみ抽出する
    const labelFields = result.fields.filter(f => f.fieldType === "Label");
    
    // それぞれのセレクトボックスにオプション追加
    const target_field1 = document.getElementById('button_mount_1');
    labelFields.forEach(field => {
      const opt = document.createElement("option");
      opt.value = field.uniqueId;  
      opt.textContent = field.caption;
      target_field1.appendChild(opt);
    });

    const target_field2 = document.getElementById('button_mount_2');
    labelFields.forEach(field => {
      console.log(field);
      const opt = document.createElement("option");
      opt.value = field.uniqueId;
      opt.textContent = field.caption;
      target_field2.appendChild(opt);
    });

    const target_field3 = document.getElementById('button_mount_3');
    labelFields.forEach(field => {
      console.log(field);
      const opt = document.createElement("option");
      opt.value = field.uniqueId;
      opt.textContent = field.caption;
      target_field3.appendChild(opt);
    });

  }

  // ---------------------------
  // ボタン表示名の復元
  // ---------------------------
  function restoreButtonLabels() {
    [1,2,3].forEach(function(n){
      var inp = document.getElementById('button_label_' + n);
      if (!inp) return;
      var saved = cfg('button_label_' + n);
      if (saved) inp.value = saved;
    });
  }

  // ---------------------------
  // ボタン表示名・配置場所 をリアルタイムに config.values へ反映
  // ---------------------------
  function bindLiveSyncForButtonSettings() {
    [1, 2, 3].forEach(function (n) {
      var labelKey = 'button_label_' + n;
      var mountKey = 'button_mount_' + n;

      var labelEl = document.getElementById(labelKey);
      if (labelEl) {
        // 入力が変わったら config.values に反映
        labelEl.addEventListener('input', function () {
          if (config.values) {
            config.values[labelKey] = labelEl.value;
          }
        });
      }

      var mountEl = document.getElementById(mountKey);
      if (mountEl) {
        // セレクト変更時に config.values に反映
        mountEl.addEventListener('change', function () {
          if (config.values) {
            config.values[mountKey] = mountEl.value;
          }
        });
      }
    });
  }


// ---------------------------
// ボタン配置場所の復元（button_mount_1〜3）
// ---------------------------
function restoreButtonMounts() {
  [1,2,3].forEach(function(n){
    const key = 'button_mount_' + n;
    const sel = document.getElementById(key);
    if (!sel) return;

    const saved = cfg(key);  // config.values[key] or config[key]
    if (!saved) return;      // 空なら何もしない（＝未設定）

    sel.value = String(saved);

    // 念のため、変更時に config.values 側も追随させておく
    sel.addEventListener('change', function(){
      if (config.values) {
        config.values[key] = sel.value;
      }
    });
  });
}

  // ---------------------------
  // 同じフィールドが他グループで既に選ばれているかチェック
  // ---------------------------
  function isFieldUsedInOtherGroups(currentGroupNo, fieldValue) {
    var used = false;

    [1, 2, 3].forEach(function (n) {
      if (n === currentGroupNo) return;

      var otherContainer = document.getElementById('group_fields_container_' + n);
      if (!otherContainer) return;

      var otherCb = otherContainer.querySelector(
        'input[type=checkbox][value="' + fieldValue + '"]'
      );
      if (otherCb && otherCb.checked) {
        used = true;
      }
    });

    return used;
  }

// フィールド一覧取得API実行
atPocket.api('apps/' + config.appId + '/fields', 'GET', {}, function (result) {
  // 成功の場合
  createSelectBox(result);          // ボタン配置場所セレクトに option 追加
  renderAllFieldCheckboxes(result); // 対象フィールドcheckbox
  restoreButtonLabels();            // ボタン表示名の復元
  bindLiveSyncForButtonSettings();  // 入力変更を config.values に同期するイベントを張る
}, function (result) {
  console.log(result);
});
})();