(() => {
  'use strict';

  // プラグインの設定を取得する
  const config = atPocket.plugin.app.getConfig();

  // ----------------------------------------------------------------
  // CSV / 配列どちらでも "field-xx" の配列にして返す
  // ----------------------------------------------------------------
  function parseCodes(raw) {
    if (!raw) return [];
    if (Array.isArray(raw)) {
      return raw.map(v => String(v).trim()).filter(Boolean);
    }
    if (typeof raw === 'string') {
      return raw
        .split(',')
        .map(v => v.trim())
        .filter(Boolean);
    }
    return [];
  }

  // ----------------------------------------------------------------
  // ボタン
  // ----------------------------------------------------------------
  function makeButton(text, onClick) {
    const btn = document.createElement('button');
    btn.type = 'button';
    btn.textContent = text;

    // サイズ
    btn.style.padding = '8px 20px';
    btn.style.fontSize = '13px';
    btn.style.fontWeight = '600';

    // レイアウト
    btn.style.display = 'inline-flex';
    btn.style.alignItems = 'center';
    btn.style.justifyContent = 'center';
    btn.style.gap = '6px';
    btn.style.border = 'none';
    btn.style.borderRadius = '999px'; // pill 型
    btn.style.cursor = 'pointer';
    btn.style.whiteSpace = 'nowrap';

    // 色・影
    btn.style.background = 'linear-gradient(90deg, #2563eb, #3b82f6)';
    btn.style.color = '#fff';
    btn.style.boxShadow = 'none';

    // アニメーション
    btn.style.transition = 'transform 0.15s ease, box-shadow 0.15s ease, opacity 0.15s ease';

    btn.addEventListener('mouseenter', () => {
      btn.style.transform = 'translateY(-1px)';
      btn.style.opacity = '0.95';
    });

    btn.addEventListener('mouseleave', () => {
      btn.style.transform = 'translateY(0)';
      btn.style.opacity = '1';
    });

    btn.addEventListener('click', e => {
      e.preventDefault();
      e.stopPropagation();
      onClick();
    });
    return btn;
  }


  // 「配置場所フィールドコード」からラベルDOMを探す
  function findMountDom(fieldCode) {
    if (!fieldCode) {
      return null;
    }
    let code = String(fieldCode).replace(/^#/, ''); // 念のため先頭 # を除去
    // 基本的に atPocket のフィールドDOMは id="field-xx" になっている想定
    const candidates = [
      code,
      'field-' + code.replace(/^field-/, ''),
      'field-field-' + code.replace(/^field-/, '')
    ];

    for (let i = 0; i < candidates.length; i++) {
      const el = document.getElementById(candidates[i]);
      if (el) {
        return el;
      }
    }

    // 見つからなければヘッダ右側あたりに出す
    return (
      document.querySelector('.record-header-right') ||
      document.querySelector('.record-header') ||
      document.body
    );
  }

  // ----------------------------------------------------------------
  // 表示制御
  // ----------------------------------------------------------------
  function hideFields(fieldCodes) {
    fieldCodes.forEach(code => {
      atPocket.app.record.setFieldShown(code, false);
    });
  }

  function showFields(fieldCodes) {
    fieldCodes.forEach(code => {
      atPocket.app.record.setFieldShown(code, true);
    });
  }

  // ----------------------------------------------------------------
  // ボタン生成（グループごと）
  // ----------------------------------------------------------------
  function injectToggleButtonForGroup(groupNo, record) {
    const idx = String(groupNo);

    // 対象フィールド群
    const targetFields = parseCodes(config['group_fields_' + idx] || '');
    if (!targetFields.length) {
      return;
    }

    // 初期状態は非表示にしておく
    hideFields(targetFields);

    const buttonLabel =config['button_label_' + idx] || '詳細を表示';
    const mountCode =config['button_mount_' + idx] || targetFields[0];

    // 「フィールドの入力欄」を取得（横に並べたい対象）
    const fieldValueEl = atPocket.app.record.getFieldElement(mountCode);

    // 「配置場所DOM」（ヘッダ or フィールド全体）
    const mountDom = findMountDom(mountCode);
    if (!mountDom) {
      return;
    }

    let isShown = false; // 現在の表示状態（false = 非表示）
    const btn = makeButton(buttonLabel, () => {
      if (!isShown) {
        // 表示する
        showFields(targetFields);
        isShown = true;
        btn.textContent = '▼ ' + buttonLabel; // 開いている状態
      } else {
        // 再度押されたら非表示に戻す
        hideFields(targetFields);
        isShown = false;
        btn.textContent = '▲ ' + buttonLabel; // 閉じている状態
      }
    });

    btn.id = 'group_toggle_button_' + idx;
    // 初期表示テキスト（閉じている状態）
    btn.textContent = '▲ ' + buttonLabel;

    // ---------------------------
    // 横並びレイアウト
    // ---------------------------
    if (fieldValueEl && fieldValueEl.parentNode) {
      // 入力欄の親要素の中に「入力欄＋ボタン」用のラッパーを作る
      const wrapper = document.createElement('div');
      wrapper.style.display = 'flex';
      wrapper.style.alignItems = 'center';
      wrapper.style.gap = '8px';

      // wrapper を挿し込んで、入力欄とボタンを中に入れる
      fieldValueEl.parentNode.insertBefore(wrapper, fieldValueEl);
      wrapper.appendChild(fieldValueEl);
      wrapper.appendChild(btn);
    } else {
      // もしうまく取れなかった場合は、従来通り末尾に付ける（ヘッダ表示など）
      mountDom.appendChild(btn);
    }
  }

// ----------------------------------------------------------------
// レコード作成/編集/詳細画面表示時に処理を実行
// ----------------------------------------------------------------
  atPocket.events.on(['app.record.create.show', 'app.record.edit.show', 'app.record.detail.show'],event => {
      // グループ1〜3を順番に処理
      injectToggleButtonForGroup(1, event.record);
      injectToggleButtonForGroup(2, event.record);
      injectToggleButtonForGroup(3, event.record);
      return event;
    }
  );
})();
