// config.js
(() => {
  'use strict';

  const config = atPocket.plugin.app.getConfig();

  const NONE_VALUE = "OPT_NONE";   // 未選択
  const BLANK_KEY  = "OPT_BLANK";  // 空(ブランク)

  let _apiResult = null;          // /fields API キャッシュ
  let _singleRadioFields = null;  // SingleSelect/Radio のフィールド配列
  let _dupGuardSuspend = 0; // 0以外なら重複ガードは何もしない

  // 保存済みのキー群から最大Nを推定し、view_count より優先して使う
  function getSavedCount() {
    const vcount = parseInt((config?.values?.view_count ?? '0'), 10) || 0;
    
    // あり得る最大Nをざっくり探索（必要なら上限を広げてOK）
    let maxN = 0;
    for (let i = 1; i <= 99; i++) {
      const hasSelect = (config?.values?.[`view_select${i}`] ?? undefined) !== undefined;
      const hasLimit  = (config?.values?.[`view_limit${i}`]  ?? undefined) !== undefined;
      if (hasSelect || hasLimit) maxN = i;
    }

    const n = Math.max(vcount, maxN);
    return n > 0 ? n : 1; // 最低1行
  }

// クリック委譲：toggle_view_limit（カタログ）と toggle_view_limit_N（各行）を一括で面倒見る
document.addEventListener('click', (ev) => {
  const t = /** @type {HTMLElement} */ (ev.target);

  // 1) 行削除ボタン（先に判定）
  const del = t.closest('button[id^="remove_view_row_"]');
  if (del) {
    const totalRows = document.querySelectorAll('select[id^="view_select"]').length;
    if (totalRows <= 1) {
      alert('表示設定は少なくとも1行は必要です。');
      return;
    }
    const row = del.closest('.config_row');
    if (row) {
      row.remove();
    }
    renumberRows();             // ID/name/見出しの再採番
    syncViewCountFromDOM();     // view_count の再同期
    return;
  }

  // 2) 表示/非表示トグル（後に判定）
  const btn = /** @type {HTMLButtonElement|null} */(
    t.closest('button[id^="toggle_view_limit"]')
  );
  if (!btn) {
    return;
  }

  //  HTMLElement に型を絞る
  let box = /** @type {HTMLElement|null} */ (null);

  const id = btn.id;
  const m  = id.match(/^toggle_view_limit_(\d+)$/);

  if (m) {
    box = /** @type {HTMLElement|null} */(
      document.getElementById(`view_limit_container_${m[1]}`)
    );
  } else if (id === 'toggle_view_limit') {
    box = /** @type {HTMLElement|null} */(
      document.getElementById('view_limit_container')
    );
  } else {
    const row = btn.closest('.config_row');
    box = /** @type {HTMLElement|null} */(
      row ? row.querySelector('[id^="view_limit_container_"]') : null
    );
  }
  if (!box) {
    return;
  }

  const cur = getComputedStyle(box).display;
  box.style.display = (cur === 'none') ? 'block' : 'none';
});

/**
* DOM上の .config_row 群を 1..N で再採番する。
* 各行の select / hidden / ボタン等の id/name も同期し、config.values にも反映。
* 削除後のギャップ解消や並び替え後に使用。
*/
function renumberRows() {
  console.log("renumberRows");
  const rows = document.querySelectorAll('.config_row'); 
  let idx = 0;

  rows.forEach(row => {
    const cap = row.querySelector('.config_caption');
    if (!cap || !/表示設定\d+/.test(cap.textContent || '')) return;

    idx += 1;
    const newIndex = idx;
    cap.textContent = `表示設定${newIndex}`;

    // select
    const selEl = row.querySelector('select[id^="view_select"]');
    if (selEl instanceof HTMLSelectElement) {
      //  今の値を保持
      const prevVal = selEl.value;

      selEl.id   = `view_select${newIndex}`;
      selEl.name = `view_select${newIndex}`;

      //  config.values にも反映
      if (config?.values) {
        config.values[`view_select${newIndex}`] = prevVal;
      }
    }

    // toggle ボタン
    const toggle = row.querySelector('button[id^="toggle_view_limit_"]');
    if (toggle instanceof HTMLButtonElement) {
      toggle.id = `toggle_view_limit_${newIndex}`;
    }

    // チェックボックスコンテナ
    const cont = row.querySelector('[id^="view_limit_container_"]');
    if (cont instanceof HTMLDivElement) {
      cont.id = `view_limit_container_${newIndex}`;
    }

    // hidden
    const hidden = row.querySelector('input[type="hidden"][id^="view_limit_values_"]');
    if (hidden instanceof HTMLInputElement) {
      const prevHidden = hidden.value;

      hidden.id   = `view_limit_values_${newIndex}`;
      hidden.name = `view_limit${newIndex}`;

      //  config.values にも反映
      if (config?.values) {
        config.values[`view_limit${newIndex}`] = prevHidden;
      }
    }

    // 削除ボタン
    const delBtn = row.querySelector('button[id^="remove_view_row_"]');
    if (delBtn instanceof HTMLButtonElement) {
      delBtn.id = `remove_view_row_${newIndex}`;
    }
  });
}

// 現在のDOMに存在する view_selectN の最大Nを返す（なければ0）
function getCurrentMaxIndex() {
  const els = document.querySelectorAll('select[id^="view_select"]');
  let max = 0;
  els.forEach(el => {
    const m = el.id.match(/^view_select(\d+)$/);
    if (m) {
      const n = parseInt(m[1], 10);
      if (Number.isFinite(n) && n > max) {
        max = n;
      }
    }
  });
  return max;
}

// DOMの行数(最大N)で view_count を同期
function syncViewCountFromDOM() {
  const viewCount = document.getElementById('view_count');
  const maxIndex = getCurrentMaxIndex();
  if (viewCount) {
    viewCount.value = String(maxIndex);
    if (config?.values) {
      config.values.view_count = viewCount.value; // ← ここを追加
    }
  }
}

/**
* 現在選択されている切替対象フィールド(#view_field)の値を安全に取得する。
* @returns {string} uniqueId 文字列（見つからない場合は空文字）
*/
function getCurrentViewFieldSafe() {
  const el = document.getElementById('view_field');
  return el instanceof HTMLSelectElement ? el.value : "";
}

/**
* config.values から key に対応するID配列を取得。配列を受け取り、
* 返却時は文字列配列に正規化する。
*/
function getSavedIds(key) {
   const v = config?.values?.[key];
   // 配列でもCSVでも「文字列」に正規化
   if (Array.isArray(v)) {
    return v.map(x => String(x));
   }
   if (typeof v === 'string' && v.trim()){
     return v.split(',').map(s => String(s.trim())).filter(Boolean);
   }
   return [];
}

/**
* select 要素の option を、BLANK/未選択 を保ちつつ指定 items で再構築する。
* allowNone=false のときは (未選択) を生成しない（Radio系のとき）。
*/
function setOption(items, element, allowNone = true) {
  let hasBlank = false, hasNone = false;
  Array.from(element.options).forEach(opt => {
    if (opt.value === BLANK_KEY) {
      if (hasBlank) {
        opt.remove(); return; 
      }
      opt.textContent = "";
      hasBlank = true;
    } else if (opt.value === NONE_VALUE) {
      if (hasNone) {
        opt.remove(); return; 
      }
      opt.textContent = "(未選択)";
      hasNone = true;
    } else {
      opt.remove();
    }
  });

  if (!hasBlank) {
    const blank = document.createElement("option");
    blank.value = BLANK_KEY;
    blank.textContent = "";
    element.insertBefore(blank, element.firstChild);
  }

  // Radio の場合は allowNone=false にする
  if (allowNone && !hasNone) {
    const none = document.createElement("option");
    none.value = NONE_VALUE;
    none.textContent = "(未選択)";
    const afterBlank = element.querySelector(`option[value="${BLANK_KEY}"]`)?.nextSibling;
    element.insertBefore(none, afterBlank || null);
  }

  items.forEach(item => {
    const opt = document.createElement("option");
    opt.value = String(item.id);
    opt.textContent = item.selectName;
    element.appendChild(opt);
  });
}

/**
* 設定行セレクト(#view_selectN)の NodeList から、有効な HTMLSelectElement の配列を返す。
* @returns {{selects: HTMLSelectElement[]}}
*/
function _getSettingSelectEls() {
  const nodes = document.querySelectorAll('select[id^="view_select"]');
  /** @type {HTMLSelectElement[]} */
  const selects = [];
  for (let i = 0; i < nodes.length; i++) {
    const el = nodes[i];
    if (!(el instanceof HTMLSelectElement)){
      continue; // ← ここで絞る
    }
    if (/^view_select\d+$/.test(el.id)){
      selects.push(el);
    }
  }
  return { selects };
} 

/**
* 表示設定行(div.config_row) を新規に生成する。
* 追加ボタン押下時や初期行生成時に使用。
* @param {number} index 生成する行番号(1始まり)
* @returns {HTMLDivElement}
*/
function _createSettingRow(index) {
  const row = document.createElement("div");
  row.className = "config_row";
  row.innerHTML = `
    <div class="config_caption">表示設定${index}</div>
    <div class="config_input">
      <!-- ヘッダ行：同じ1行で横並び -->
      <div class="view-row-head" style="display:flex; align-items:center; gap:6px; flex-wrap:wrap;">
        <select id="view_select${index}" name="view_select${index}" class="view-select">
          <option value="OPT_BLANK"></option>                 <!-- ← 見た目を真っ白にするプレースホルダ -->
        </select>
        <button type="button" id="toggle_view_limit_${index}">▼ 表示フィールド</button>
        <button type="button" id="remove_view_row_${index}">-削除</button>
      </div>

      <!-- チェックボックス群：次の行で全幅使用 -->
      <div id="view_limit_container_${index}" class="chk-wrap" style="display:none; margin-top:6px;"></div>

      <!-- 保存用 hidden（Gridの子にしない） -->
      <input type="hidden" id="view_limit_values_${index}" name="view_limit${index}" value="">
    </div>
  `;
  return row;
}

//指定した input[type=hidden] の値を CSV として取り出し、文字列配列にする
function getLiveCsvIdsById(id) {
   const el = /** @type {HTMLInputElement|null} */(document.getElementById(id));
   if (!el){
    return [];
   }
   const s = (el.value || '').trim();
   return s ? s.split(',').map(x => String(x.trim())).filter(Boolean) : [];
}

//切替対象フィールド(#view_field)の変更時ハンドラ。
function onViewFieldChanged(ev) {
  const select = ev.currentTarget;
  if (!(select instanceof HTMLSelectElement)) return;
  if (!_apiResult) return;

  const newValue = select.value;
  const oldValue = config?.values?.view_field || "";
  const isNew = oldValue !== newValue;

  // 候補を再構築
  createSelectSelect(_apiResult.fields, newValue);

  // 新規切替なら全行をリセット
  if (isNew) {
    // ① 各行の select をリセット
    document.querySelectorAll("select[id^='view_select']").forEach(sel => {
      if (sel instanceof HTMLSelectElement) {
        sel.value = BLANK_KEY;
        if (config?.values) {
          config.values[sel.id] = BLANK_KEY;
        }
      }
    });

    // ② 各行のチェックボックスをリセット（カタログは残す）
    document.querySelectorAll("[id^='view_limit_container_']").forEach(cont => {
      // 行ごとのチェックボックスを全部外す
      cont.querySelectorAll('input[type="checkbox"]').forEach(cb => {
        cb.checked = false;
      });

      // hidden の値も空にする
      const hidden = cont.querySelector('input[type="hidden"][id^="view_limit_values_"]');
      if (hidden instanceof HTMLInputElement) {
        hidden.value = "";
        if (config?.values) {
          config.values[hidden.name] = "";
        }
      }
    });

    // カタログ（toggle_view_limit, view_limit）はそのまま残す
  }

  // 保存値更新
  if (config?.values) {
    config.values.view_field = newValue;
  } else {
    config.values = { view_field: newValue };
  }
}


// 行ごとのチェックボックスを描画し、hidden(name=view_limitN)に同期する
function renderRowCheckboxes(fields, index, selectedIds = []) {
  const container = document.getElementById(`view_limit_container_${index}`);
  if (!container) return;

  const allowedIds = getLiveCsvIdsById('view_limit_values');
  const allowedStr = new Set(allowedIds.map(String));
  const selectedStr = new Set(selectedIds.map(String));

  const pool = allowedIds.length > 0
    ? fields.filter(f => allowedStr.has(String(f.uniqueId)))
    : [];

  // hidden 作成
  let hidden = document.getElementById(`view_limit_values_${index}`);
  if (!(hidden instanceof HTMLInputElement)) {
    hidden = document.createElement("input");
    hidden.type = "hidden";
    hidden.id   = `view_limit_values_${index}`;
    hidden.name = `view_limit${index}`;
    container.insertAdjacentElement("afterend", hidden);
  }
  hidden.value = [...selectedStr].join(",");

  // 描画リセット
  container.innerHTML = "";

  if (pool.length === 0) {
    const note = document.createElement("div");
    note.style.color = "#666";
    note.textContent =
      "※「切替対象フィールド」で選択した項目のみ表示されます。";
    container.appendChild(note);
    return;
  }

  // チェックボックス描画
  pool.forEach(field => {
    const id = String(field.uniqueId);
    const label = document.createElement("label");
    label.style.display = "block";

    const cb = document.createElement("input");
    cb.type = "checkbox";
    cb.value = id;
    cb.checked = selectedStr.has(id);

    cb.addEventListener("change", () => {
      const ids = Array.from(container.querySelectorAll('input[type="checkbox"]:checked'))
        .map(c => /** @type {HTMLInputElement} */ (c).value);
      hidden.value = ids.join(",");
      config.values[hidden.name] = hidden.value;
    });

    label.appendChild(cb);
    label.appendChild(document.createTextNode(" " + (field.caption ?? field.name ?? field.label)));
    container.appendChild(label);
  });

  // --- 全選択／全解除ボタン ---
  if (pool.length > 0) {
    const btnWrap = document.createElement("div");
    btnWrap.style.marginTop = "6px";
    btnWrap.style.display = "flex";
    btnWrap.style.gap = "6px";

    // 全選択
    const checkAllBtn = document.createElement("button");
    checkAllBtn.type = "button";
    checkAllBtn.textContent = "全選択";
    checkAllBtn.addEventListener("click", () => {
      container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
        /** @type {HTMLInputElement} */(cb).checked = true;
      });
      hidden.value = Array.from(
        container.querySelectorAll('input[type="checkbox"]:checked')
      ).map(c => /** @type {HTMLInputElement} */(c).value).join(",");
      config.values[hidden.name] = hidden.value;
    });

    // 全解除（この行だけ対象！）
    const uncheckAllBtn = document.createElement("button");
    uncheckAllBtn.type = "button";
    uncheckAllBtn.textContent = "全解除";
    uncheckAllBtn.addEventListener("click", () => {
      container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
        /** @type {HTMLInputElement} */(cb).checked = false;
      });
      hidden.value = "";
      config.values[hidden.name] = "";
    });
    uncheckAllBtn.dataset.scope = "row";

    btnWrap.appendChild(checkAllBtn);
    btnWrap.appendChild(uncheckAllBtn);
    container.appendChild(btnWrap);
  }
}

// 既存の renderCatalogCheckboxes を下のように末尾だけ変更
function renderCatalogCheckboxes(fields, selectedIds = []) {
  const container = document.getElementById("view_limit_container");
  if (!container){
    return;
  }

  let hidden = /** @type {HTMLInputElement|null} */(document.getElementById("view_limit_values"));
  if (!hidden) {
    hidden = document.createElement('input');
    hidden.type = 'hidden';
    hidden.id = 'view_limit_values';
    hidden.name = 'view_limit';
    container.insertAdjacentElement('afterend', hidden);
  }

  const selectedStr = new Set(selectedIds.map(String));

  container.innerHTML = "";
  fields.forEach(field => {
    const label = document.createElement("label");
    label.style.display = "block";
    const cb = document.createElement("input");
    cb.type = "checkbox";
    cb.value = String(field.uniqueId);
    cb.checked = selectedStr.has(String(field.uniqueId));
    label.appendChild(cb);
    label.appendChild(document.createTextNode(" " + (field.caption ?? field.name ?? field.label)));
    container.appendChild(label);
  });

   // --- 全選択／全解除ボタン（カタログ） ---
  if (fields.length > 0) {
    const btnWrap = document.createElement("div");
    btnWrap.style.marginTop = "6px";
    btnWrap.style.display = "flex";
    btnWrap.style.gap = "6px";

    const checkAllBtn = document.createElement("button");
    checkAllBtn.type = "button";
    checkAllBtn.textContent = "全選択";
    checkAllBtn.addEventListener("click", () => {
      container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
        /** @type {HTMLInputElement} */(cb).checked = true;
      });
      container.dispatchEvent(new Event("change")); // hidden 更新 & 全行再描画
    });

    const uncheckAllBtn = document.createElement("button");
    uncheckAllBtn.type = "button";
    uncheckAllBtn.textContent = "全解除";
    uncheckAllBtn.addEventListener("click", () => {
      container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
        /** @type {HTMLInputElement} */(cb).checked = false;
      });
      container.dispatchEvent(new Event("change")); // hidden 更新 & 全行再描画
    });
    uncheckAllBtn.dataset.scope = "catalog";

    btnWrap.appendChild(checkAllBtn);
    btnWrap.appendChild(uncheckAllBtn);
    container.appendChild(btnWrap);
  }

  hidden.value = selectedIds.join(',');

 //  カタログが変わったら hidden を更新 → 全行を再描画
container.onchange = () => {
  const ids = Array.from(container.querySelectorAll('input[type="checkbox"]:checked'))
    .map((c) => String((/** @type {HTMLInputElement} */(c)).value));

  hidden.value = ids.join(',');

  // ここで config.values も更新しておくとより確実
  if (config?.values) {
    config.values.view_limit = hidden.value;
  }

  // 全行更新
  if (_apiResult) rerenderAllRowsFromCatalog(_apiResult.fields);
};

}

// renderCatalogCheckboxes の外に1回だけ書けばOK
document.addEventListener("click", (ev) => {
  const btn = ev.target;
  if (btn instanceof HTMLButtonElement && btn.dataset.scope === "catalog" && btn.textContent === "全解除") {
    const container = document.getElementById("view_limit_container");
    const hidden = document.getElementById("view_limit_values");
    if (!container || !(hidden instanceof HTMLInputElement)) return;
    // カタログ側チェックを全部外す
    container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
      cb.checked = false;
    });

    // カタログ hidden を空にする
    hidden.value = "";
    if (config?.values) {
      config.values.view_limit = "";   //  config.values も更新
    }

    // 各行 hidden も全部空にする
    document.querySelectorAll('[id^="view_limit_values_"]').forEach(h => {
      if (h instanceof HTMLInputElement) {
        h.value = "";
        if (config?.values) {
          config.values[h.name] = "";  //  ここで保存値も更新
        }
      }
    });

    // 全行再描画
    if (_apiResult) rerenderAllRowsFromCatalog(_apiResult.fields);
  }
});

// 追加：全行をカタログ選択に合わせて再描画
function rerenderAllRowsFromCatalog(fields) {
  const allowedIds = getLiveCsvIdsById('view_limit_values');

  document.querySelectorAll('[id^="view_limit_container_"]').forEach(el => {
    const m = el.id.match(/^view_limit_container_(\d+)$/);
    if (!m) return;
    const i = parseInt(m[1], 10);

    const hidden = document.getElementById(`view_limit_values_${i}`);
    if (!(hidden instanceof HTMLInputElement)) return;

    let selected = getSavedIds(`view_limit${i}`);

    if (allowedIds.length === 0) {
      //  hidden は空にするけど renderRowCheckboxes に [] を渡さない
      hidden.value = "";
      renderRowCheckboxes(fields, i, selected);
      return;
    }

    renderRowCheckboxes(fields, i, selected);
  });
}

// ===== 切替対象フィールド（チェック群）を全行へ描画 =====
function createSelectLimit(result) {
  const excludeIds = [-101, -104, -105, -106, -107, -119];
  const fields = result.fields.filter(f => !excludeIds.includes(f.fieldId));

  // カタログの保存値を復元して描画
  const savedCatalog = getSavedIds('view_limit');
  renderCatalogCheckboxes(fields, savedCatalog);

  // 保存済み行数ぶん描画＆復元
  const total = getSavedCount();
  for (let i = 1; i <= total; i++) {
    const selected = getSavedIds(`view_limit${i}`); //  保存値を復元
    renderRowCheckboxes(fields, i, selected);
  }
}

function renderExistingRowCheckboxes(fields) {
  // DOMに存在する view_limit_container_N を片っ端から描画（復元）
  document.querySelectorAll('[id^="view_limit_container_"]').forEach(el => {
    const m = el.id.match(/^view_limit_container_(\d+)$/);
    if (!m) {
      return;
    }
    const i = parseInt(m[1], 10);
    const selected = getSavedIds(`view_limit${i}`);
    renderRowCheckboxes(fields, i, selected);
  });
}

/**
 * 切替フィールド (#view_field) のセレクトボックスを生成・初期化する。
 */
function createSelectBox(result) {
  const fields = result.fields.filter(f =>
    f.fieldType === "SingleSelect" || f.fieldType === "Radio"
  );
  _singleRadioFields = fields;

  const selectEl = document.getElementById('view_field');
  if (!selectEl || !(selectEl instanceof HTMLSelectElement)) {
    console.warn("#view_field が見つからない");
    return;
  }

  // optionクリア
  while (selectEl.options.length > 0) {
    selectEl.remove(0);
  }

  // プレースホルダ
  const none = document.createElement("option");
  none.value = "";
  none.textContent = "(未選択)";
  selectEl.appendChild(none);

  // 候補追加
  fields.forEach(field => {
    const opt = document.createElement("option");
    opt.value = field.uniqueId;
    opt.textContent = field.caption || field.name || field.label || ("ID:" + field.uniqueId);
    selectEl.appendChild(opt);
  });

  // 保存値を反映
  if (config?.values?.view_field) {
    selectEl.value = String(config.values.view_field);
  }

  selectEl.addEventListener("change", onViewFieldChanged);
}


// ===== 切替値の候補を各 view_selectN に配布（既存選択を保持） =====
function createSelectSelect(fields, value) {
  const { selects } = _getSettingSelectEls();

  const target = (value && value !== NONE_VALUE && value !== BLANK_KEY)
    ? fields.find(f => String(f.uniqueId) === String(value) && Array.isArray(f.selectItems))
    : null;

  const items = target ? target.selectItems : [];
  const allowNone = target ? (target.fieldType === "SingleSelect") : true; // ここ

  _dupGuardSuspend++;
  try {
    for (let selectEl of selects) {
      while (selectEl.options.length > 1) selectEl.remove(1);
      setOption(items, selectEl, allowNone);
      selectEl.value = BLANK_KEY;
      bindDuplicateGuardTo(selectEl);
    }
  } finally {
    _dupGuardSuspend--;
  }
}


// ===== 追加ボタン =====
(function setupAddButton() {
  const addBtn = /** @type {HTMLButtonElement|null} */(document.getElementById("add_setting_btn"));
  const formRoot = document.querySelector(".config_form");
  const addRow = addBtn ? addBtn.closest(".config_row") : null;
  const viewCount = /** @type {HTMLInputElement|null} */(document.getElementById("view_count"));
  if (!addBtn || !formRoot || !addRow || !viewCount) return;

  addBtn.addEventListener("click", function () {
  syncConfigValuesFromDOM();   //  ここでまず同期
  const newIndex = getCurrentMaxIndex() + 1;
  const row = _createSettingRow(newIndex);
  formRoot.insertBefore(row, addRow);

  if (_apiResult) {
    // 新規行だけ描画
    renderRowCheckboxes(_apiResult.fields, newIndex, []);
    const currentViewField = getCurrentViewFieldSafe();
    if (currentViewField) {
      const selectEl = document.getElementById(`view_select${newIndex}`);
      if (selectEl) {
        const target = _apiResult.fields.find(
        f => f.uniqueId === currentViewField && Array.isArray(f.selectItems)
        );
        const items = target ? target.selectItems : [];
        //  SingleSelect のときだけ未選択を許可
        const allowNone = target ? (target.fieldType === "SingleSelect") : true;
        setOption(items, selectEl, allowNone);
        selectEl.value = ""; // 追加直後は空
        bindDuplicateGuardTo(selectEl);
      }
    }
  }

  // view_count だけ更新
  syncViewCountFromDOM();
});

})();

/**
* 現在のDOM状態から config.values を最新値で埋め直す。
* 提出前や行追加前に実行して、保存値の取りこぼしを防ぐ。
*/
function syncConfigValuesFromDOM() {
  if (!config?.values) config.values = {};

  // --- 切り替え対象フィールド (view_field) ---
  const vf = document.getElementById("view_field");
  if (vf instanceof HTMLSelectElement) {
    config.values.view_field = vf.value;
  }

  // --- 表示設定 (view_selectN) ---
  document.querySelectorAll('select[id^="view_select"]').forEach(sel => {
    config.values[sel.id] = sel.value;
  });

  // --- 各行 hidden (view_limitN) ---
  document.querySelectorAll('input[type="hidden"][id^="view_limit_values_"]').forEach(h => {
    if (h instanceof HTMLInputElement) {
      config.values[h.name] = h.value;  // h.name = view_limitN
    }
  });

  // --- カタログ hidden (view_limit) ---
  const catalogHidden = document.getElementById("view_limit_values");
  if (catalogHidden instanceof HTMLInputElement) {
    config.values.view_limit = catalogHidden.value;
  }
}

// ===== 初期化：保存された行数に合わせて行DOMを用意 =====
(function ensureInitialRows() {
  const total = getSavedCount();
  const viewCount = /** @type {HTMLInputElement|null} */(document.getElementById("view_count"));
  const formRoot = document.querySelector(".config_form");
  const addRow = document.getElementById("add_setting_btn")?.closest(".config_row") || null;
  if (!viewCount || !formRoot || !addRow) {
    return;
  }

  // 表示設定1はHTMLにある前提。2〜Nを作成
  for (let i = 2; i <= total; i++) {
    const row = _createSettingRow(i);
    formRoot.insertBefore(row, addRow);
  }
  //  行数を保存値に合わせる
  viewCount.value = String(total);

  // DOMをもとに再同期
  syncViewCountFromDOM();
})();

// ===== フィールド一覧の取得 → 復元＆描画 =====
atPocket.api(`apps/${config.appId}/fields`, 'GET', {}, function (result) {
  _apiResult = result;

  // 1) 切替フィールド候補 + 保存値復元
  createSelectBox(result);

  // 2) 各表示設定行の切替値候補を配布（保持/復元）
  createSelectSelect(result.fields, getCurrentViewFieldSafe());

  // 3) 各行のチェック群を保存値で描画
  createSelectLimit(result);

  rerenderAllRowsFromCatalog(result.fields);
  renderExistingRowCheckboxes(result.fields);

  // 4) 表示設定N の選択値も保存値で復元
  const total = getSavedCount();
  for (let i = 1; i <= total; i++) {
    const saved = config?.values?.[`view_select${i}`] ?? "";
    const sel = document.getElementById(`view_select${i}`);
    if (sel instanceof HTMLSelectElement && saved) {
      sel.value = saved;
    }
  }

  syncViewCountFromDOM();
  bindDuplicateGuards();
}, function(err) {
  console.log(err);
});


// 登録（保存）ボタン押下時に DOM → config.values を同期
document.addEventListener("submit", (ev) => {
  const form = ev.target;
  if (form instanceof HTMLFormElement && form.classList.contains("config_form")) {
    syncViewCountFromDOM();        // 行数を反映
    syncConfigValuesFromDOM();     // 各 view_selectN / view_limitN を反映
  }
});

// セレクトの比較キー：空値は __NONE__ に正規化
function valueKeyOf(selectEl) {
  const raw = (selectEl.value || "").trim();
  if (raw === NONE_VALUE) {
    return NONE_VALUE;  // (未選択)
  }
  return raw;// 実値
}

// --- 重複ガード（変更時に即チェック & ロールバック） ---
function bindDuplicateGuardTo(selectEl) {
  if (!(selectEl instanceof HTMLSelectElement)) return;
  if (selectEl.dataset.dupGuardBound === "1") return;

  selectEl.dataset.dupGuardBound = "1";

  //  フォーカスした時点で「直前値」を保存しておく
  selectEl.addEventListener("focus", () => {
    selectEl.dataset.prevRaw = (selectEl.value || "");
    selectEl.dataset.prevKey = valueKeyOf(selectEl);
  });

  selectEl.addEventListener("change", () => {
    if (_dupGuardSuspend) return;

    const curKey = valueKeyOf(selectEl);
    //  空(ブランク)は重複チェック対象外
    if (curKey !== BLANK_KEY) {
      const all = Array.from(document.querySelectorAll('select[id^="view_select"]'));
      const dup = all.find(s => s !== selectEl && valueKeyOf(s) === curKey);
      if (dup) {
        const msg =
          curKey === NONE_VALUE ? '「未選択」' : '同じ値';
        alert(`表示設定の選択肢が重複しています。元に戻します。`);

        //  focus 時点で保存した直前値に戻す
        const prevRaw = selectEl.dataset.prevRaw ?? "";
        selectEl.value = prevRaw;
        selectEl.focus();
        return;
      }
    }
    //  重複が無かった場合にだけ、直前値を更新する
    selectEl.dataset.prevKey = curKey;
    selectEl.dataset.prevRaw = (selectEl.value || "");
  });
}

//すべての #view_selectN に重複ガードをバインドする。
function bindDuplicateGuards() {
  const els = document.querySelectorAll('select[id^="view_select"]');
  Array.from(els).forEach(bindDuplicateGuardTo);
}

})();
